/* Copyright (C) 2004 Nicolas DUPEUX

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>

#include <stdio.h>
#include <string.h>



#ifdef DEBUG
#define DBG_MSG(format, ...)   fprintf(stderr, format, ## __VA_ARGS__)
#else
#define DBG_MSG(format, args...)
#endif

struct MemoryStruct {
	char *memory;
	size_t size;
};

size_t
WriteMemoryCallback(void *ptr, size_t size, size_t nmemb, void *data)
{
	register int realsize = size * nmemb;
	struct MemoryStruct *mem = (struct MemoryStruct *)data;

	mem->memory = (char *)realloc(mem->memory, mem->size + realsize + 1);
	if (mem->memory) {
		memcpy(&(mem->memory[mem->size]), ptr, realsize);
		mem->size += realsize;
		mem->memory[mem->size] = 0;
	}
	return realsize;
}

typedef struct st_mysql_field {
	char *name;                   /* Name of column */
	char *table;                  /* Table of column if column was a field */
	char *org_table;              /* Org table name if table was an alias */
	char *db;                     /* Database for table */
	char *def;                    /* Default value (set by mysql_list_fields) */
	unsigned long length;         /* Width of column */
	unsigned long max_length;     /* Max width of selected set */
	unsigned int flags;           /* Div flags */
	unsigned int decimals;        /* Number of decimals in field */
	//enum enum_field_types type;   /* Type of field. Se mysql_com.h for types */
} MYSQL_FIELD;  

enum mysql_option { MYSQL_OPT_CONNECT_TIMEOUT, MYSQL_OPT_COMPRESS,
			MYSQL_OPT_NAMED_PIPE, MYSQL_INIT_COMMAND,
			MYSQL_READ_DEFAULT_FILE, MYSQL_READ_DEFAULT_GROUP,
			MYSQL_SET_CHARSET_DIR, MYSQL_SET_CHARSET_NAME,
			MYSQL_OPT_LOCAL_INFILE};


typedef char** MYSQL_ROW;

typedef unsigned int MYSQL_FIELD_OFFSET;
typedef unsigned int MYSQL_ROW_OFFSET;

typedef struct {
	MYSQL_FIELD **fields;
	MYSQL_FIELD_OFFSET field_offset;
	MYSQL_FIELD_OFFSET num_fields;
	unsigned int num_rows;
	MYSQL_ROW *rows;
	MYSQL_ROW_OFFSET row_offset;
	unsigned long **lengths;
} MYSQL_RES;


typedef struct {
	char *host;
	char *db;
	char *user;
	char *passwd;
	char *login_id;
	CURL *curl;
	struct HttpPost *formpost;
	struct HttpPost *lastptr;
	struct MemoryStruct hResponse;
	unsigned int errno;
	char *error;
	MYSQL_RES *last_result;
} MYSQL;

typedef unsigned long long my_ulonglong;

typedef char my_bool;

void mysql_free_result(MYSQL_RES *result);

void __attribute__ ((constructor)) mysql_wa_init(void)
{
	/* Initialise the curl lybrary */
	curl_global_init(CURL_GLOBAL_ALL);
}

void __attribute__ ((destructor)) mysql_wa_fini(void)
{
	/* cleanup the curl library */
	curl_global_cleanup();
}

char* next_line(char *text)
{
	while ((*text!=0) && (*text!='\n')) {
		++text;
	}
	if (*text) ++text;
	return text;
}

char* copy_line(char *text)
{
	int len=0;
	char *ptr=text;
	char *line;

	while ((*ptr!=0) && (*ptr!='\n')) {
		++len;
		++ptr;
	}

	line=(char*)malloc((len+1)*sizeof(char));
	
	memcpy(line,text,len*sizeof(char));
	line[len]='\0';

	return line;
}

char* read_field(char **data,char delim,unsigned long *len)
{
	char *ptr=*data;
	char *line;

	*len=0;

	// Read up to the next delim
	while ((*ptr!=0) && (*ptr!=delim) && (*ptr!='\n')) {
		++(*len);
		++ptr;
	}

	if (*len==0) {(*data)++; return NULL; }

	line=(char*)malloc(((*len)+1)*sizeof(char));
	memcpy(line,*data,(*len)*sizeof(char));
	line[*len]='\0';

	if ((*ptr=='\n') || (*ptr==delim)) (*len)++;

	*data+=(*len);

	return line;
}

my_ulonglong mysql_affected_rows(MYSQL *mysql)
{
	DBG_MSG("mysql_affected_rows()\n");
	return mysql->last_result->num_rows;
}

my_ulonglong mysql_insert_id(MYSQL *mysql)
{
	DBG_MSG("mysql_insert_id()\n");
	return 0;
}

int mysql_select_db(MYSQL *mysql, const char *db)
{
	DBG_MSG("mysql_select_db()\n");
	free(mysql->db);
	mysql->db=strdup(db);

	return 0;
}

unsigned int mysql_errno(MYSQL *mysql)
{
	DBG_MSG("mysql_errno()\n");
	return mysql->errno;
}

const char *mysql_error(MYSQL *mysql)
{
	DBG_MSG("mysql_error()\n");
	return mysql->error;
}

int mysql_ping(MYSQL *mysql)
{
	DBG_MSG("mysql_ping()\n");
	return 0;
}

unsigned int mysql_num_fields(MYSQL_RES *result)
{
	DBG_MSG("mysql_num_fields() = %i\n",result->num_fields);
	return result->num_fields;
}

MYSQL_ROW mysql_fetch_row(MYSQL_RES *result)
{
	DBG_MSG("mysql_fetch_row()\n");
	//
	if (result->row_offset < result->num_rows) {
					result->row_offset++;
					return result->rows[result->row_offset-1];
	} else 
					return NULL;
}

unsigned long *mysql_fetch_lengths(MYSQL_RES *result)
{
	DBG_MSG("mysql_fetch_lengths()\n");
	
	return result->lengths[result->row_offset-1];
}

MYSQL_RES *mysql_store_result(MYSQL *mysql)
{
	MYSQL_RES *res=(MYSQL_RES*)calloc(sizeof(MYSQL_RES),1);
	char *data;
	int i;
	int j;
	unsigned long len=0;

	DBG_MSG("mysql_store_result()\n");

	if (mysql->errno) return NULL;

	// Parse the output
	data=next_line(mysql->hResponse.memory);

	if (memcmp(data,"ERROR",strlen("ERROR"))==0) {
		data=next_line(data);
		mysql->errno=strtol(data,&data,10);
		mysql->error=read_field(&data,',',&len);
		return NULL;
	}
	data=next_line(data);
	// number of fields in the result
	res->num_fields=strtol(data,&data,10);
	data++;
	res->fields=(MYSQL_FIELD**)malloc(res->num_fields*sizeof(MYSQL_FIELD*));

	// read description of each field
	for (i=0; i < res->num_fields ; i++) {
		res->fields[i]=(MYSQL_FIELD*)malloc(sizeof(MYSQL_FIELD));
		res->fields[i]->name=read_field(&data,',',&len);
		res->fields[i]->table=read_field(&data,',',&len);
		res->fields[i]->length=strtol(data,&data,10);
		res->fields[i]->max_length=res->fields[i]->length;
		res->fields[i]->flags=strtol(data,&data,10);
		data=next_line(data);
	}

	// number of row in the result
	res->num_rows=strtol(data,&data,10);
	data++;
	res->rows=(MYSQL_ROW*)malloc(res->num_rows*sizeof(MYSQL_ROW));
	res->lengths=(unsigned long**)malloc(res->num_rows*sizeof(unsigned long*));

	// read rows
	for (i=0; i < res->num_rows; i++) {
		res->rows[i]=(MYSQL_ROW)malloc(res->num_fields*sizeof(char*));
		res->lengths[i]=(unsigned long*)malloc(res->num_fields*sizeof(unsigned long));
		for (j=0; j < res->num_fields; j++) {
			res->rows[i][j]=read_field(&data,',',&(res->lengths[i][j]));
		}
	}

	free(mysql->hResponse.memory);
	mysql->hResponse.memory=NULL;
	mysql->hResponse.size=0;

	mysql->last_result=res;

	mysql->errno=0;
	if (mysql->error) free(mysql->error);
	mysql->error=strdup("");

	return res;
}

MYSQL_RES *mysql_use_result(MYSQL *mysql)
{
	// implemented just like store_result

	DBG_MSG("mysql_use_result()\n");

	return mysql_store_result(mysql);
}

int mysql_real_query(MYSQL *mysql, const char *query, unsigned long length)
{
	unsigned long len;
	CURLcode res;
	char *data;

	DBG_MSG("mysql_real_query(%s)\n",query);

	/* database to access */
	curl_formadd(&mysql->formpost,
		&mysql->lastptr,
		CURLFORM_COPYNAME, "db",
		CURLFORM_COPYCONTENTS, mysql->db,
		CURLFORM_END);

	/* We want to submit a query */
	curl_formadd(&mysql->formpost,
		&mysql->lastptr,
		CURLFORM_COPYNAME, "fct",
		CURLFORM_COPYCONTENTS, "query",
		CURLFORM_END);

	/* And here is the query */
	curl_formadd(&mysql->formpost,
		&mysql->lastptr,
		CURLFORM_COPYNAME, "query",
		CURLFORM_COPYCONTENTS, query,
		CURLFORM_END);

	/* set the POST header */
	curl_easy_setopt(mysql->curl, CURLOPT_HTTPPOST, mysql->formpost);

	curl_easy_setopt(mysql->curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
	curl_easy_setopt(mysql->curl, CURLOPT_WRITEDATA, (void *)&mysql->hResponse);

	res=curl_easy_perform(mysql->curl);

	curl_formfree(mysql->formpost);
	mysql->formpost=NULL;
	mysql->lastptr=NULL;

	if( (mysql->hResponse.size > 0) && (res == 0) ) {
		DBG_MSG("%s\n",mysql->hResponse.memory);
	} else {
		fprintf(stderr,"error\n");
		return CR_SERVER_LOST;
	}

	// Read the beginning of the result to get
	// error code
	//
	// Ignore version for the moment
	data=next_line(mysql->hResponse.memory);

	if (memcmp(data,"ERROR",strlen("ERROR"))==0) {
		data=next_line(data);
		mysql->errno=strtol(data,&data,10);
		data++;
		if (mysql->error) free(mysql->error);
		mysql->error=read_field(&data,',',&len);
		free(mysql->hResponse.memory);
		mysql->hResponse.memory=NULL;
		mysql->hResponse.size=0;
	} else {
		mysql->errno=0;
		if (mysql->error) free(mysql->error);
		mysql->error=strdup("");
	}
	data=next_line(data);

	return mysql->errno;
}


MYSQL *mysql_init(MYSQL *mysql)
{
	DBG_MSG("mysql_init()\n");

	if (mysql==NULL) {
		return (MYSQL*)calloc(sizeof(MYSQL),1);
	} else {
		memset(mysql,'\0',sizeof(MYSQL));
		return mysql;
	}
}

char *mysql_get_server_info(MYSQL *mysql)
{
	const char *sql="SELECT VERSION()";
	char *version;
	MYSQL_RES* result;
	MYSQL_ROW  row;

	DBG_MSG("mysql_get_server_info()\n");
	
	if (mysql_real_query(mysql,sql,strlen(sql))) {
		return NULL;
	}

	// Analyse the result
	result=mysql_store_result(mysql);
	row = mysql_fetch_row(result);

	version=strdup(row[0]);

	mysql_free_result(result);

	return version;
}

my_bool mysql_change_user(MYSQL *mysql, const char *user, const char
		*passwd, const char *db)
{
	DBG_MSG("mysql_change_user()\n");

	mysql->user=user ? strdup(user) : strdup("");
	mysql->passwd=passwd ? strdup(passwd) : strdup("");

	/* login and password */
	if (mysql->login_id) free(mysql->login_id);
	mysql->login_id = (char*)malloc(
			(strlen(mysql->user)+strlen(mysql->passwd)+2)
			*sizeof(char));
	sprintf(mysql->login_id,"%s:%s",mysql->user,mysql->passwd);
	curl_easy_setopt(mysql->curl, CURLOPT_USERPWD, mysql->login_id);

	return 1;
}

MYSQL *mysql_real_connect(MYSQL *mysql, const char *host,
		const char *user, const char *passwd, const char *db,
		unsigned int port, const char *unix_socket,
		unsigned long client_flag)
{
	char *version;

	DBG_MSG("mysql_real_connect()\n");

	mysql->curl = curl_easy_init();

	if (host) mysql->host=strdup(host);
	if (db) mysql->db=strdup(db);

	/* url */
	curl_easy_setopt(mysql->curl, CURLOPT_URL, host);

	/* login and password */
	mysql_change_user(mysql,user,passwd,db);

	// We ask the server version to test the connexion
	version=mysql_get_server_info(mysql);
	if (!version)
		// Can't get the version
		return NULL;
	free(version);

	return mysql;
}

void mysql_free_result(MYSQL_RES *result)
{
	int i,j;

	DBG_MSG("mysql_free_result()\n");

	for (j=0; j < result->num_fields; j++) {
		free(result->fields[j]->name);
		free(result->fields[j]->table);
		free(result->fields[j]);
	}
	free(result->fields);

	for (i=0; i < result->num_rows ; i++) {
		for (j=0; j < result->num_fields; j++) {
			free(result->rows[i][j]);
		}
		free(result->rows[i]);
		free(result->lengths[i]);
	}
	free(result->rows);
	free(result->lengths);

	free(result);
}

void mysql_close(MYSQL *mysql)
{
	DBG_MSG("mysql_close()\n");

	free(mysql->host);
	free(mysql->db);
	free(mysql->user);
	free(mysql->passwd);
	free(mysql->login_id);
	free(mysql->error);
	curl_easy_cleanup(mysql->curl);

	return;
}

MYSQL_RES *mysql_list_fields(MYSQL *mysql, const char *table, const char *wild)
{
	const char *sql="SHOW COLUMNS FROM %s LIKE '%s'";
	char *query;
	MYSQL_RES *result;

	DBG_MSG("mysql_list_fields(%s)\n",table);
		

	if (!table) return NULL;
	wild=wild ? wild : "%";

	query=(char*)malloc((strlen(sql)+strlen(table))*sizeof(char));
	sprintf(query,sql,table,wild);

	mysql_real_query(mysql,query,strlen(query));
	free(query);
	result=mysql_store_result(mysql);

	return result;
}

unsigned long mysql_thread_id(MYSQL *mysql)
{
	DBG_MSG("mysql_thread_id() = 69\n");
	return 69;
}

my_ulonglong mysql_num_rows(MYSQL_RES *result)
{

	DBG_MSG("mysql_num_rows() = %i\n",result->num_rows);

	return result->num_rows;
}

void mysql_data_seek(MYSQL_RES *result, my_ulonglong offset)
{

	DBG_MSG("mysql_data_seek()\n");

	result->row_offset=offset;
}

unsigned int mysql_field_count(MYSQL *mysql)
{

	DBG_MSG("mysql_field_count()\n");

	return mysql->last_result->num_fields;
}

MYSQL_FIELD_OFFSET mysql_field_seek(MYSQL_RES *result,
		MYSQL_FIELD_OFFSET offset)
{
	MYSQL_FIELD_OFFSET previous=result->field_offset;

	DBG_MSG("mysql_field_seek()\n");

	result->field_offset=offset;
	return previous;
}

MYSQL_FIELD_OFFSET mysql_field_tell(MYSQL_RES *result)
{
	DBG_MSG("mysql_field_tell()\n");
	return result->field_offset;
}

MYSQL_FIELD *mysql_fetch_fields(MYSQL_RES *result)
{
	DBG_MSG("mysql_fetch_fields()\n");
	
	return *(result->fields);
}

MYSQL_FIELD *mysql_fetch_field(MYSQL_RES *result)
{
	DBG_MSG("mysql_fetch_field()\n");
	
	if (result->field_offset < result->num_fields) {
		result->field_offset++;
		return result->fields[result->field_offset-1];
	}
	else
		return NULL;
}

int mysql_options(MYSQL *mysql, enum mysql_option option, const char *arg)
{
	DBG_MSG("mysql_options()\n");

	return 0;
}

const char *mysql_character_set_name(MYSQL *mysql)
{
	CURLcode res;
	unsigned long len;
	char *data;
	char *charset;

	DBG_MSG("mysql_character_set_name()\n");

	curl_formadd(&mysql->formpost,
		&mysql->lastptr,
		CURLFORM_COPYNAME, "fct",
		CURLFORM_COPYCONTENTS, "character_set_name",
		CURLFORM_END);

	/* set the POST header */
	curl_easy_setopt(mysql->curl, CURLOPT_HTTPPOST, mysql->formpost);

	curl_easy_setopt(mysql->curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
	curl_easy_setopt(mysql->curl, CURLOPT_WRITEDATA, (void *)&mysql->hResponse);

	res=curl_easy_perform(mysql->curl);

	curl_formfree(mysql->formpost);
	mysql->formpost=NULL;
	mysql->lastptr=NULL;

	if( (mysql->hResponse.size > 0) && (res == 0) ) {
		fprintf(stderr,"%s\n",mysql->hResponse.memory);
	}

	data=next_line(mysql->hResponse.memory);
	charset=read_field(&data,'\n',&len);

	free(mysql->hResponse.memory);
	mysql->hResponse.memory=NULL;
	mysql->hResponse.size=0;

	return charset;
}


